package ga.core.logging;

import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IPopulation;

/**
 * Interface for a logger used by GAs. This provides callbacks on important
 * events during the process.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public interface IGALogger<T extends IIndividual<T>> {
  /**
   * Notifies about the start of the evaluation.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void evaluationStarted();

  /**
   * Notifies about the initialization of the population.
   * 
   * @param generation
   *          The current generation.
   * @param population
   *          The population that has been initialized.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void populationInitiated(int generation, IPopulation<T> population);

  /**
   * Notifies about individuals that have been selected.
   * 
   * @param generation
   *          The current generation.
   * @param list
   *          The list of selected individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void individualsSelected(int generation, IndividualList<T> list);

  /**
   * Notifies about individuals that have been crossed.
   * 
   * @param generation
   *          The current generation.
   * @param list
   *          The list of crossed individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void individualsCrossed(int generation, IndividualList<T> list);

  /**
   * Notifies about individuals that have been mutated.
   * 
   * @param generation
   *          The current generation.
   * @param list
   *          The list of mutated individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void individualsMutated(int generation, IndividualList<T> list);

  /**
   * Notifies about individuals that have been inserted into the population.
   * 
   * @param generation
   *          The current generation.
   * @param list
   *          The list individuals that have been inserted.
   * @param population
   *          The population where the individuals have been inserted.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void individualsInserted(int generation, IndividualList<T> list,
      IPopulation<T> population);

  /**
   * Notifies about an individual that has been selected for evaluation.
   * 
   * @param generation
   *          The current generation.
   * @param individual
   *          The selected individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void individualSelectedForEvaluation(int generation, T individual);

  /**
   * Notifies about an individual that has been evaluated.
   * 
   * @param generation
   *          The current generation.
   * @param individual
   *          The evaluated individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void individualEvaluated(int generation, T individual);

  /**
   * Notifies about the population that has been evaluated.
   * 
   * @param generation
   *          The current generation.
   * @param population
   *          The population that has been evaluated.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void allIndividualsEvaluated(int generation, IPopulation<T> population);

  /**
   * Notifies about the termination of the algorithm. The logger must shut down,
   * too.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void exit();
}
